/** @mainpage MC Programmer's Guide
 *  
 * <center>                                                                            
 * MC Programmer's Guide                                                  
 * FETK was written by Michael Host.<br>                                               
 * Additional contributing authors are listed in the code documentation.               
 * </center>                                              
 * 
 * <hr width="100%">                                                                   
 *  @section toc   Table of Contents
 *  <ul>
 *  <li> @ref intro
 *  <li> @ref style
 *  <li> @ref api
 *    <ul>
 *    <li> <a href="modules.html">Modules</a>
 *    <li> <a href="annotated.html">Class list</a>
 *    <li> <a href="functions.html">Class members</a>
 *    <li> <a href="globals.html">Class methods</a>
 *    </ul>
 *  <li> @ref license
 *  </ul>
 * 
 *  <p>
 * <i>
 * NOTE:
 * </i>
 * This documentation provides information about the programming interface
 * provided by the MC software and a general guide to linking to the MC
 * libraries.  Information about installation, configuration, and general usage
 * can be found in the <a href="../../index.html">User's Guide</a>.
 * <hr>
 * 
 *  \section intro Introduction
 * 
 * <p>
 * MC
 * (Manifold Code) was written by 
 * <a href="http://www.scicomp.ucsd.edu/~mholst/">Michael Holst</a>
 * at Caltech and UC San Diego, primarily as a finite element kernel for
 * <a href="http://www.fetk.org">FETK</a>
 * (the Finite Element TookKit).
 * 
 *  \section formalism Clean OO C Formalism
 * 
 *  <p>
 *  MC was written in
 *  <a href="../../cooc.html">Clean OO C</a>,
 *  which is a self-imposed disciplined coding formalism for 
 *  producing object-oriented ANSI/Standard C which can be
 *  compiled with either a C or C++ compiler.
 *  We will briefly describe the formalism here
 *  (borrowing from N. Baker's nice description in the APBS documentation).
 *
 *  <p>
 *  Clean OO C formalism requires that all public data is
 *  enclosed in structures which resemble C++ classes.  These structures and
 *  member functions are then declared in a public header file which provides a
 *  concise description of the interface (or API) for the class.
 *  Private functions and
 *  data are included in private header files (or simply the source code files
 *  themselves) which are not visible generally visible (data encapsulation).
 *  When using the library, the
 *  user only sees the public header file and the compiled library and is
 *  therefore (hopefully) oblivious to the private members and functions.  Each
 *  class is also equipped with a constructor and destructor function which is
 *  responsible for allocating and freeing any memory required by the
 *  instatiated objects.
 * 
 *  <p>
 *  Public data members are enclosed in C structures which
 *  are visible to the library user.  Public member functions are generated by
 *  mangling the class and function names and passing a pointer to the
 *  object on which the member function is supposed to act.  For example, a
 *  public member function with the C++ declaration 
 *    <pre>
 *   public double Foo::bar(int i, double d)
 *   </pre>
 * would be declared as
 *   <pre>
 *   VEXTERNC double Foo_bar(Foo *thee, int i, double d)
 *   </pre>
 * where <code>VEXTERNC</code> is a compiler-dependent macro, the underscore
 * <code>_</code> replaces the C++ double-colon <code>::</code>, and
 * <code>thee</code> replaces the <code>this</code> variable implicit in all
 * C++ classes.  Since they do not appear in public header files, private
 * functions could be declared in any format pleasing to the user, however, the
 * above declaration convention should generally be used for both public and
 * private functions.  Within the source code, the public and private function
 * declarations/definitions are prefaced by the macros <code>VPUBLIC</code> and
 * <code>VPRIVATE</code>, respectively.  These are macros which reduce global
 * name pollution, similar to encapsulating private data withing C++ classes.
 *  
 * <p>
 * The only C++ functions not explicitly covered by the above declaration
 * scheme are the constructors (used to allocate and initialize class data
 * members) and destructors (used to free allocated memory).  These are
 * declared in the following fashion:  a constructor with the C++ declaration
 *    <pre>
 *    public void Foo::Foo(int i, double d)
 *    </pre>
 * would be declared as
 *     <pre>
 *     VEXTERNC Foo* Foo_ctor(int i, double d)
 *     </pre>
 * which returns a pointer to the newly constructed <code>Foo</code> object.
 * Likewise, a destructor declared as
 *     <pre>
 *     public void Foo::~Foo()
 *     </pre>
 * in C++ would be
 *     <pre>
 *     VEXTERNC void Foo_dtor(Foo **thee)
 *     </pre>
 * in Clean OO C.
 * <p>
 * Finally, inline functions in C++ are simply treated as macros in Clean OO C
 * and declared/defined using "#define" statements in the public
 * header file.
 * See any of the MC header files for more examples on the Clean OO C
 * formalism.
 * 
 *  \section style Coding Style
 * 
 * <p>
 * The best (and most entertaining) description of a C coding style that is
 * very close to what I use in MC is that described by Linus Torvalds in his
 * "CodingStyle" file in the Linux kernel sources (usually found in the file
 * "/usr/src/linux/Documentation/CodingStyle" on any Linux box).  He describes
 * a coding style that is modular, completely documented (but in a spartan way),
 * and very practical.
 *
 * <p>
 * Below are some additional notes on the coding style
 * I use in MC beyond what Torvalds describes.
 * These additional guidelines
 * are mostly concerned with being compatible with the Clean OO C dialect,
 * giving an object-oriented look and feel to MC.
 * <ul>
 * <li>
 *     Tabs and spaces ==> Probably the most important rule: NO TABS!
 *     <p>
 *          Use ONLY SPACES, NO TABS, in source code.
 *          When indenting a code block, ALWAYS USE EXACT 4 SPACES,
 *          no more, no less.  While this single anal rule seems
 *          excessive, in my experience it is the single most useful
 *          code formatting guideline one can impose, in terms of
 *          producing code written by many different developers that
 *          can be read and understood quickly by other developers
 *          using the same convention.  If four spaces (rather than
 *          two or three) forces you to use more than 80 columns for
 *          nesting loops/etc, then your routine is too complex to
 *          be read by someone else anyway and it should be split
 *          into two or more routines.
 *     <p>
 * <li>
 *     ALLCAPS     ==> A "#define"d constant or a macro.  Examples:
 *     <pre>
 *     "#define" VTRUE    1
 *     "#define" VFALSE   0
 *     "#define" VABS(x)  ((x) >= 0 ? (x) : -(x))
 *     </pre>
 * <li>
 *     UpperMixed  ==> Class name, or class constructor or destructor.
 *                     Examples:
 *     <p>
 *     <pre>
 *     C++:     class Mesh { ... };
 *              Mesh(void) { ... }
 *              ~Mesh(void) { ... }
 * 
 *     Clean C: typedef struct Mesh { ... } Mesh;
 *              void Mesh_ctor(Mesh *thee) { ... }
 *              void Mesh_dtor(Mesh *thee) { ... }
 *     </pre>
 *            NOTE:  The "this" pointer is implicitly used in C++. We
 *                   simulate this coding style with the "thee" pointer
 *                   in Clean C.  By using a name different than "this", 
 *                   we continue to have a legal C++ program.
 *     <p>
 * <li>
 *     lowerMixed  ==> A class member function or generic function.  Examples:
 *     <p>
 *     <pre>
 *     C++:    Class Mesh {
 *                ...
 *                void print(void) { ... }
 *                ...
 *             }
 *             void Mesh::plot(void) { ... }
 * 
 *     Clean C: void Mesh_print(Mesh *thee) { ... }
 *              void Mesh_plot(Mesh *thee) { ... }
 *     </pre>
 * <li>
 *     _private data ==> Pre-fixed with an underscore, such as:  vint _dim.
 *     <p>
 *           I don't always adhere to this, but it helps make
 *           it clear when you are doing something unsafe with
 *           private data that you should have written an accessor
 *           member function to handle.  (UPDATE: I no longer allow
 *           this practice in MC, because symbols beginning with
 *           an underscore often conflict with internal compiler
 *           symbols in GCC and other ANSI-C compilers.)
 * <li>
 *     Datatypes ==> Use only standard or compatible datatypes.
 *     <p>
 *           Beyond the stuctures used to define classes, only standard
 *           datatypes are used as primitive types in MC, such as
 *           char, int, float, and double.
 *     <p>
 * <li>
 *     Core structures ==> Refer to headers vel.h and ves.h in the GEM library
 *                         for examples.
 *     <p>
 * <li>
 *     Other comments ==> We basically adhere to the GNU coding conventions,
 *                        except that Richard Stallman recommends (in his GNU
 *                        Coding Standards paper) writing blocks as follows:
 *     <pre>
 *     if (cond)
 *     {
 *     }
 *     </pre>
 *                        whereas I find the following usually more readable:
 *     <pre>
 *     if (cond) {
 *     }
 *     </pre>
 *                        However, for complete routines I do follow
 *                        Richard's approach:
 *     <pre>
 *     void func(void)
 *     {
 *         ...stuff...
 *     }
 *     </pre>
 *                        If something is very simple then I break all rules:
 *     <pre>
 *     int myId(void) { return id; }
 *     </pre>
 * </ul>
 * 
 * @section api Application programming interface documentation
 * <p>
 * The API documentation for this code was generated by
 * <a href="http://www.doxygen.org">doxygen</a>.  You can either view the API
 * documentation by using the links at the top of this page, or the slight
 * re-worded/re-interpreted list below:
 *    <ul>
 *    <li> <a href="modules.html">Class overview</a>
 *    <li> <a href="annotated.html">Class declarations</a>
 *    <li> <a href="functions.html">Class members</a>
 *    <li> <a href="globals.html">Class methods</a>
 *    </ul>
 * 
 * @section license License
 *
 @verbatim

    MC = < Manifold Code >
    Copyright (C) 1994-- Michael Holst

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
   
 @endverbatim
 */
